const fs = require("fs");
const path = require("path");
const sizeLimits = require("./size_limits.json");
const { readdirSyncRecursive, mkdirSyncRecursive } = require("./resources/utils.js");
const { execSync } = require("child_process");

const INPUT_DIR = path.join(__dirname, "in");
const OUTPUT_DIR = path.join(__dirname, "out");
const RESOURCE_DIR = "resources";

process.stdout.write("\033c");
mkdirSyncRecursive(OUTPUT_DIR);

// Compress video to a specified file size (kB)
const compressVideo = (input, output, desiredSize) => {
	// Calculate the duration of the video in seconds with ffprobe
	const duration = parseFloat(
		execSync(
			`ffprobe -v error -show_entries format=duration -of default=noprint_wrappers=1:nokey=1 "${input}"`
		)
	);
	// Calculate the desired bitrate based on the filesize we want and the duration of the video.
	// For some reason the calculation results in file sizes much smaller than desired, so we
	// multiply by a magic number (7) to try and bring it back to where we want it
	const bitrate = Math.floor(desiredSize / duration) * 7;
	// Ffmpeg 2 pass for desired file size (https://trac.ffmpeg.org/wiki/Encode/H.264#twopass)
	execSync(`
		ffmpeg -y -i "${input}" -c:v libx264 -b:v ${bitrate}k -pass 1 -an -f mp4 -loglevel panic /dev/null && \\
		ffmpeg -i "${input}" -c:v libx264 -b:v ${bitrate}k -pass 2 -an -loglevel panic -y "${output}"
	`);
};

// Iterate through each mp4 ad file recursively
readdirSyncRecursive(INPUT_DIR).forEach(file => {
	if (!file.endsWith(".mp4")) return;

	// Extract ad unit dimensions from file name (300x250 for example)
	const adDimensions = /\d{2,}(X|x)\d{2,}/.exec(file)[0].toLowerCase();
	const adWidth = adDimensions.split("x")[0];
	const adHeight = adDimensions.split("x")[1];

	// Make folder for ad unit (named after the dimensions)
	let dirBase = file.substr(file.indexOf(INPUT_DIR) + INPUT_DIR.length);
	dirBase = dirBase.substr(0, dirBase.lastIndexOf("/"));
	const dir = path.join(OUTPUT_DIR, dirBase, adDimensions);
	mkdirSyncRecursive(dir);

	// Compress video using ffmpeg and move it into ad folder
	const destinationPath = path.join(dir, "base.mp4");
	const desiredFileSize = sizeLimits.find(ad => ad.dimensions === adDimensions)
		.subload;
	compressVideo(file, destinationPath, desiredFileSize);

	// Copy html template into folder
	// And replace placeholder height/width values with real ones
	let html = fs.readFileSync(path.join(RESOURCE_DIR, "template.html"), "utf8");
	html = html.replace(/%%WIDTH%%/g, adWidth).replace(/%%HEIGHT%%/g, adHeight);
	fs.writeFileSync(path.join(dir, "index.html"), html);

	console.log(`${dirBase}/${adDimensions} complete!`);
});

// Clean up ffmpeg log files
fs.readdirSync("./")
	.filter(file => file.startsWith("ffmpeg2pass"))
	.forEach(file => fs.unlinkSync(file));

console.log("\n=================\nAll ads complete!\n=================\n");
